#![doc = include_str!("../README.md")]

use std::{
    fs::{create_dir_all, File},
    io::Write,
    path::PathBuf,
    time::{Duration, Instant},
};

use indicatif::{MultiProgress, ProgressBar, ProgressStyle};

pub type FnTimed<T> = Box<dyn Fn(ProgressBar) -> TimeWithValue<T>>;

pub struct LabeledFnTimed<T> {
    pub label: String,
    pub func: FnTimed<T>,
}

impl<T> From<(String, FnTimed<T>)> for LabeledFnTimed<T> {
    fn from((label, func): (String, FnTimed<T>)) -> Self {
        LabeledFnTimed { label, func }
    }
}

#[derive(Debug, Clone)]
/// hold information about the benchmark to run
pub struct Bencher {
    /// the number of times each bench should run, the higher this number the lower the variance
    nb_measurements: usize,
    /// the name of the benchmark
    name: String,

    file: Option<PathBuf>,
    append: bool,
}

impl Bencher {
    pub fn new(nb_measurements: usize) -> Self {
        Self {
            nb_measurements,
            name: "".to_string(),
            file: None,
            append: false,
        }
    }

    /// add a name to a bencher
    pub fn with_name(&self, name: impl ToString) -> Self {
        Self {
            nb_measurements: self.nb_measurements,
            name: name.to_string(),
            file: self.file.clone(),
            append: self.append,
        }
    }

    /// add a file to a bencher
    pub fn with_file(&self, path: PathBuf) -> Self {
        Self {
            nb_measurements: self.nb_measurements,
            name: self.name.clone(),
            file: Some(path),
            append: self.append,
        }
    }
    /// append to file
    pub fn append(&self) -> Self {
        Self {
            nb_measurements: self.nb_measurements,
            name: self.name.clone(),
            file: self.file.clone(),
            append: true,
        }
    }
    /// overwrite file
    pub fn overwrite(&self) -> Self {
        Self {
            nb_measurements: self.nb_measurements,
            name: self.name.clone(),
            file: self.file.clone(),
            append: false,
        }
    }

    fn dump(&self, label: &str, times: &[u128], append: bool) {
        let output = label! { label: label, name: self.name, times: times }.to_string();
        if let Some(path) = &self.file {
            create_dir_all(path.parent().unwrap()).unwrap_or_else(|_| {
                panic!(
                    "{}: could not create parent directory of log file '{}'",
                    self.name,
                    path.to_str().unwrap()
                )
            });
            let mut f = File::options()
                .write(true)
                .append(append)
                .truncate(!append)
                .create(true)
                .open(path)
                .unwrap_or_else(|_| {
                    panic!(
                        "{}: could not open log file '{}'",
                        self.name,
                        path.to_str().unwrap()
                    )
                });
            writeln!(&mut f, "{}", output).unwrap_or_else(|_| {
                panic!(
                    "{}: could not write to log file '{}'",
                    self.name,
                    path.to_str().unwrap()
                )
            });
        } else {
            println!("{}", output);
        }
    }

    /// benchmark multiple pieces of code
    ///
    /// - benches is a list of label-f pairs:
    ///     - label: an additional label to differentiate similar but different things to benchmark in the
    ///     same bencher
    ///     - f: the piece of code to run and measure
    /// - the measurements will be printed to STDOUT or to file as JSON
    /// - the result of the last run will be returned
    #[allow(clippy::type_complexity)]
    pub fn bench_multiple_and_return<O>(&self, benches: Vec<LabeledFnTimed<O>>) -> Option<O> {
        let style = ProgressStyle::with_template(
            "[{elapsed_precise}] {bar:40.cyan/blue} {pos:>10}/{len:10} {msg}",
        )
        .unwrap()
        .progress_chars("##-");

        let mpb = MultiProgress::new();

        let pb_main = mpb.add(ProgressBar::new(benches.len() as u64).with_style(style.clone()));
        pb_main.set_message(self.name.clone());

        let mut res = None;
        let mut append = self.append;
        for LabeledFnTimed { label, func } in benches.iter() {
            let pb =
                mpb.add(ProgressBar::new(self.nb_measurements as u64).with_style(style.clone()));
            pb.set_message(label.to_string());

            let mut times = vec![];
            for _ in 0..self.nb_measurements {
                let TimeWithValue { t, v } = func(pb.clone());
                res = Some(v);
                times.push(t.as_nanos());

                pb.inc(1);
            }
            pb_main.inc(1);
            pb.finish_and_clear();

            self.dump(label, &times, append);
            append = true;
        }

        pb_main.finish_with_message(format!("{} done", pb_main.message()));

        res
    }

    /// see [`Self::bench_multiple_and_return`]
    #[allow(clippy::type_complexity)]
    pub fn bench_multiple<O>(&self, benches: Vec<LabeledFnTimed<O>>) {
        self.bench_multiple_and_return(benches);
    }

    /// see [`Self::bench_multiple_and_return`]
    pub fn bench_and_return<O>(&self, label: impl ToString, f: FnTimed<O>) -> Option<O> {
        self.bench_multiple_and_return(vec![LabeledFnTimed {
            label: label.to_string(),
            func: f,
        }])
    }

    /// see [`Self::bench_multiple_and_return`]
    pub fn bench<O>(&self, label: impl ToString, f: FnTimed<O>) {
        self.bench_multiple_and_return(vec![LabeledFnTimed {
            label: label.to_string(),
            func: f,
        }]);
    }
}

#[macro_export]
macro_rules! closure {
    ($pb:ident, $( $body:stmt );* $(;)?) => {
        Box::new(move |$pb: indicatif::ProgressBar| { $( $body )* }) as plnk::FnTimed<_>
    };
    ($( $body:stmt );* $(;)?) => {
        Box::new(move |_| { $( $body )* }) as plnk::FnTimed<_>
    };
}

#[macro_export]
macro_rules! label {
    ( $( $key:ident : $value:expr ),* $(,)? ) => {{
        let mut parts = Vec::new();
        $(
            parts.push(format!("{}: {:?}", stringify!($key), $value));
        )*
        &format!("{{{}}}", parts.join(", "))
    }};
}

#[derive(Debug, Clone)]
pub struct TimeWithValue<T> {
    pub t: Duration,
    pub v: T,
}

/// measure the time it takes to do something, and return the result
///
/// # Example
/// ```rust
/// # use plnk::{timeit, TimeWithValue};
/// let TimeWithValue{ v: res, .. } = timeit(|| 1 + 2);
/// assert_eq!(res, 3);
/// ```
pub fn timeit<F, O>(f: F) -> TimeWithValue<O>
where
    F: Fn() -> O,
{
    let start_time = Instant::now();
    let v = f();
    let t = Instant::now().duration_since(start_time);
    TimeWithValue { t, v }
}
