#!/usr/bin/env python
import argparse
from typing import Tuple

import plot
import multi_bar

import json


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        "gplt",
        description="A general plotter script in Python",
        formatter_class=argparse.RawTextHelpFormatter,
    )

    subparsers = parser.add_subparsers(title="subcommands", dest="subcommand")

    plot_parser = subparsers.add_parser(
        "plot",
        help="Simple plot supporting multiple graphs and custom styling.",
        formatter_class=argparse.RawTextHelpFormatter,
    )
    plot_parser.add_argument("graphs", type=str, nargs="?", help=f"the list of graphs to plot\n\n{plot.HELP}")
    plot_parser.add_argument("--graphs-file", "-f", type=str, help="an alternate path to the graphs")
    plot_parser.add_argument("--title", "-t", type=str, help="the title of the plot")
    plot_parser.add_argument("--x-label", "-x", type=str, help="the x label of the plot")
    plot_parser.add_argument("--y-label", "-y", type=str, help="the y label of the plot")
    plot_parser.add_argument("--x-scale", "-X", type=str, choices=["linear", "log"], default="linear", help="the x scale of the plot")
    plot_parser.add_argument("--y-scale", "-Y", type=str, choices=["linear", "log"], default="linear", help="the y scale of the plot")
    plot_parser.add_argument("--x-lim", type=float, nargs=2, help="limit the x axis of the plot")
    plot_parser.add_argument("--y-lim", type=float, nargs=2, help="limit the y axis of the plot")
    plot_parser.add_argument("--fullscreen", action="store_true")
    plot_parser.add_argument("--save", "-s", type=str, help="a path to save the figure to")
    plot_parser.add_argument("--aspect-ratio", type=int, nargs=2, default=[16, 9], help="the aspect ratio of the saved image")
    plot_parser.add_argument("--dpi", type=int, default=500, help="the DPI of the save image")

    multi_bar_parser = subparsers.add_parser(
        "multi-bar",
        help="Multi-bar plot.",
        formatter_class=argparse.RawTextHelpFormatter,
    )
    multi_bar_parser.add_argument("data", type=str, nargs="?", help=f"the actual data to show in a multibar plot\n\n{multi_bar.HELP}")
    multi_bar_parser.add_argument("--data-file", "-f", type=str, help="an alternate path to the data")
    multi_bar_parser.add_argument("--title", "-t", type=str, help="the title of the multibar plot")
    multi_bar_parser.add_argument("--label", "-l", type=str, help="the measurement label of the multibar plot")
    multi_bar_parser.add_argument("--fullscreen", action="store_true")
    multi_bar_parser.add_argument("--save", "-s", type=str, help="a path to save the figure to")
    multi_bar_parser.add_argument("--aspect-ratio", type=int, nargs=2, default=[16, 9], help="the aspect ratio of the saved image")
    multi_bar_parser.add_argument("--dpi", type=int, default=500, help="the DPI of the save image")

    args = parser.parse_args()
    if args.subcommand is None:
        print("nothing to do: see `--help`")
        exit(0)

    if args.subcommand == "plot":
        if args.graphs_file is not None:
            print(f"loading graphs from `{args.graphs_file}`")
            with open(args.graphs_file, "r") as graphs_file:
                graphs = json.load(graphs_file)
        else:
            graphs = json.loads(args.graphs)

        plot.plot(
            graphs,
            args.title,
            args.x_label,
            args.y_label,
            plot_layout="constrained" if args.fullscreen else None,
            x_scale=args.x_scale,
            y_scale=args.y_scale,
            x_lim=args.x_lim,
            y_lim=args.y_lim,
            save=args.save,
            save_aspect_ratio=args.aspect_ratio,
            save_dpi=args.dpi,
        )
    elif args.subcommand == "multi-bar":
        if args.data_file is not None:
            print(f"loading data from `{args.data_file}`")
            with open(args.data_file, "r") as data_file:
                data = json.load(data_file)
        else:
            data = json.loads(args.data)

        groups, measurements = multi_bar.extract(data)

        plot_layout = "constrained" if args.fullscreen else None

        multi_bar.multi_bar(
            groups,
            measurements,
            args.title,
            args.label,
            plot_layout=plot_layout,
            save=args.save,
            save_aspect_ratio=args.aspect_ratio,
            save_dpi=args.dpi,
        )
