# GPLT - a general plotter script in Python

## Installation
```bash
make install
```

## Usage
> :bulb: **Note**  
> [Nushell](https://www.nushell.sh/) has been used as the shell for the examples.  
> Data, such as `$graphs` or `$data` is written in the
> [NUON data format](https://crates.io/crates/nuon) and calls to `gplt` use the
> external argument spreading syntax, i.e. `external ...[arg1, arg2]`.

### Simple plots
```bash
let graphs = [
    {
        name: "Alice",
        points: [
            [ x, y, e ];
            [ 1, 1143, 120 ],
            [ 2, 1310, 248 ],
            [ 4, 1609, 258 ],
            [ 8, 1953, 343 ],
            [ 16, 2145, 270 ],
            [ 32, 3427, 301 ]
        ],
        style: {},  # optional, see section below
    },
    {
        name: "Bob",
        points: [
            [ x, y, e ];
            [ 1, 2388, 374 ],
            [ 2, 2738, 355 ],
            [ 4, 3191, 470 ],
            [ 8, 3932, 671 ],
            [ 16, 4571, 334 ],
            [ 32, 4929, 1094 ]
        ]
        style: {},  # optional, see section below
    },
]
```
```bash
gplt plot ...[
    --title "TITLE"
    --x-label "X LABEL"
    --y-label "Y LABEL"
    --x-scale "linear"
    --y-scale "linear"
    --fullscreen
    ($graphs | to json)
]
```
![plot.png](assets/plot.png)

### Multi-bar plots
```bash
let data = {
    age: {
        alice: 31,
        bob: 28,
        charlie: 44,
    },
    height: {
        alice: 1.50,
        bob: 1.82,
        charlie: 1.65,
    },
    weight: {
        alice: 65.3,
        bob: 98.1,
        charlie: 68.7,
    },
}
```
```bash
gplt multi-bar ...[
    --title "TITLE"
    --label "LABEL"
    --fullscreen
    ($data | to json)
]
```
![multi_bar.png](assets/multi_bar.png)

### a concrete example: PLNK benchmarks
When using the [PLNK](https://gitlab.isae-supaero.fr/a.stevan/plnk) benchmark framework, rows of
data of the shape `record<label: string, name: string, times: list<float>>` are generated, e.g. the
NDJSON file below, which we'll call `data.ndjson` in the following
```json
{ "label": "x = 1", "name": "measure 1", "times": [ 62.33530583467004, 13.116892798969348, 200.30948986467558, 86.58000013917714, 15.134191510752103, 195.46370761223523, 36.7036294821508, 60.93627835956281, 35.79444091447257, 15.95545341499609 ] }
{ "label": "x = 2", "name": "measure 1", "times": [ 264.8746025442256, 33.94014510866023, 32.863403302526955, 128.954106790958, 60.103782258920226, 86.09669711939888, 63.57355250750817, 179.5507236842757, 16.622189982341094, 220.71197740587903 ] }
{ "label": "x = 3", "name": "measure 1", "times": [ 341.5420107480995, 199.58101184443728, 140.23099455283406, 137.63116077129348, 54.51748386780794, 10.189550756159232, 249.1212795090168, 301.51256213060606, 331.1047080094466, 219.7392895325493 ] }
{ "label": "x = 4", "name": "measure 1", "times": [ 231.91818809003377, 361.5768661556012, 155.4144463915178, 135.25818395037612, 287.97267936578004, 277.0485022968556, 285.7175344304758, 229.56060462076255, 412.27229594871494, 242.39878269146178 ] }
{ "label": "x = 5", "name": "measure 1", "times": [ 229.1286727461793, 36.05966790878941, 217.3306944760557, 465.828539314587, 466.93086984298975, 365.64715832114223, 342.00335623760503, 393.01972457664476, 126.68932417932488, 273.6437320003236 ] }
{ "label": "x = 6", "name": "measure 1", "times": [ 464.46280851954515, 525.8317328526753, 178.5525909419073, 179.2769936462735, 128.33493476241134, 505.3568650201965, 180.70658544813267, 400.99030181163187, 239.42376527132532, 350.1694860281751 ] }
{ "label": "x = 7", "name": "measure 1", "times": [ 541.0078269235072, 298.44337331453715, 295.8985400052819, 59.224860070041416, 92.84043383088809, 134.42252194210585, 166.59281627253472, 168.970502555747, 132.14435249487408, 273.7068239424026 ] }
{ "label": "x = 8", "name": "measure 1", "times": [ 420.86729548827657, 179.96184758975429, 561.1519877773519, 383.480494645818, 223.39065295700405, 462.3845608732607, 406.42556023468654, 390.26037733412045, 222.70110771492955, 349.9322805558609 ] }
{ "label": "x = 9", "name": "measure 1", "times": [ 388.44389317107965, 262.8617876710674, 408.52799531933357, 479.109880149684, 220.8002206809629, 351.47394244035974, 476.69950435805464, 637.1350294941114, 325.1988300638635, 300.1857305972319 ] }
{ "label": "x = 10", "name": "measure 1", "times": [ 128.50647098850794, 332.74504956274944, 363.5056254329104, 118.93034826526119, 358.57229415900827, 184.3045472089404, 445.6813014912866, 681.2517501030742, 678.2084346311494, 462.74046962195325 ] }
{ "label": "x = 11", "name": "measure 1", "times": [ 234.873194956317, 282.16938157403024, 470.1474465747976, 392.8303252446165, 358.0760233936245, 634.719874692147, 240.00508922063938, 345.91232071104514, 781.2532780812651, 213.53483734522175 ] }
{ "label": "x = 12", "name": "measure 1", "times": [ 365.7719981851251, 591.3910258736885, 444.94567729973323, 547.096836728914, 357.4018985024443, 256.4590981746625, 761.0437745052194, 597.7897438146892, 385.6363136797271, 475.3989408385458 ] }
{ "label": "x = 13", "name": "measure 1", "times": [ 347.51950729744686, 298.2161069387033, 174.92322250040218, 341.74070167253535, 351.8620138920042, 588.37294840278, 252.51691978485314, 314.0961140808313, 500.00005864864534, 783.417446015071 ] }
{ "label": "x = 14", "name": "measure 1", "times": [ 922.0866764517559, 294.4609407757316, 205.02570279832216, 595.5094316034636, 367.12816192719725, 634.4850899960727, 508.45973249862607, 447.38192852286477, 275.3996110510375, 673.5604876792385 ] }
{ "label": "x = 15", "name": "measure 1", "times": [ 969.3770489673472, 449.80415054301983, 514.0897875104, 839.5531001568673, 434.51157338452276, 690.0185101918808, 833.5217097872853, 956.3836179189233, 295.85908384060855, 269.7812611644785 ] }
{ "label": "x = 1", "name": "measure 2", "times": [ 38.8473674848816, 136.20318117698997, 145.56032858312955, 75.33625505396732, 65.26876659844099, 123.01920707543364, 6.6411474696095425, 11.479613730271408, 121.4899746662293, 61.22000983532488 ] }
{ "label": "x = 2", "name": "measure 2", "times": [ 80.49510613436398, 130.49809702686508, 51.33940524709127, 105.1070886352, 159.69706966974493, 149.73937921465003, 118.85652240938771, 47.48567388707895, 110.11066964375355, 142.89681336157 ] }
{ "label": "x = 3", "name": "measure 2", "times": [ 253.69788742978452, 281.887525567412, 209.86009836657948, 270.18653427709853, 50.335802360188595, 272.475172588027, 59.444392533139144, 29.98183892969703, 217.0680219644828, 116.21323143578053 ] }
{ "label": "x = 4", "name": "measure 2", "times": [ 308.31257673305083, 260.6860004451179, 324.19873372936763, 239.8916088171298, 135.78054819392014, 144.9195872891682, 143.78674440042855, 306.74294167895573, 82.08983105017148, 145.66145232560172 ] }
{ "label": "x = 5", "name": "measure 2", "times": [ 197.9213056350696, 331.09372032873466, 139.225853317775, 159.48176210709602, 355.30839639563334, 235.08502884188226, 446.3266977775306, 321.7496792097118, 381.8393313911904, 149.5914746393344 ] }
{ "label": "x = 6", "name": "measure 2", "times": [ 312.23941810189336, 374.92137863972823, 398.81286369739996, 481.4884261253473, 393.09898997443054, 336.16891955771337, 384.0937426863384, 286.76530467860374, 504.28892722333705, 363.74084476817944 ] }
{ "label": "x = 7", "name": "measure 2", "times": [ 439.9398605701669, 483.4454110541442, 454.4422630994168, 703.207925293674, 720.778754707711, 501.2546314344956, 572.6307212264065, 537.1053314980023, 634.1869024110074, 635.0381822053357 ] }
{ "label": "x = 8", "name": "measure 2", "times": [ 801.4588805282879, 609.6683836745918, 824.7078899249589, 931.3120429548942, 881.8039412315444, 852.8781832846842, 652.1877871284722, 724.1242832576986, 844.9635765225987, 657.2474976052615 ] }
{ "label": "x = 9", "name": "measure 2", "times": [ 731.1257347527966, 992.2801271519542, 856.4602343164237, 862.3949367913029, 887.4012568761214, 896.6118940175816, 835.1107709814473, 833.1742549911953, 966.0669210650902, 978.193012826573 ] }
{ "label": "x = 10", "name": "measure 2", "times": [ 1029.4697960441008, 1317.6567785778814, 1301.8317980833697, 1242.649577982329, 1369.4535117894918, 1141.5547890667835, 1251.2718486549431, 1152.4652629645698, 1197.3071881606252, 1338.2944108845054 ] }
{ "label": "x = 11", "name": "measure 2", "times": [ 1706.5593224161992, 1602.335705906727, 1456.520055920673, 1639.6130399863614, 1498.6394890428367, 1805.7208439986891, 1476.4331769848075, 1722.4330323639888, 1469.4218213381434, 1393.975659269363 ] }
{ "label": "x = 12", "name": "measure 2", "times": [ 2059.797187461384, 1763.0999820441934, 2029.2757106111299, 2231.267041522453, 1963.9124127557336, 1861.2153482702124, 2153.637078071648, 2119.3150148007726, 1975.4708438157322, 2077.5587609698478 ] }
{ "label": "x = 13", "name": "measure 2", "times": [ 2661.560270532897, 2429.5019671066425, 2496.163933142116, 2500.6778483097364, 2690.8524846222235, 2522.0652229191273, 2327.16411647266, 2355.387848841374, 2723.5832275125713, 2293.6716488165507 ] }
{ "label": "x = 14", "name": "measure 2", "times": [ 3225.070568404471, 3072.246315551287, 2998.2351804874766, 2905.73723689757, 2781.1829767320964, 3279.2638527865197, 2935.3598458006486, 3105.9932227710383, 2855.041399052157, 3138.1685472919585 ] }
{ "label": "x = 15", "name": "measure 2", "times": [ 3535.747948720741, 3677.582714199909, 3646.5720795939305, 3725.942851188326, 3767.7730406890073, 3643.164538688472, 3900.9714827849148, 3882.1454021270533, 3380.4218045891434, 3653.2220330590703 ] }
```

First, we'll load the data using the standard library of Nushell
```bash
use std formats [ "from ndjson" ]

let data = open data.ndjson
```

Then we can compute some statistics about the data, e.g. the mean and standard deviation of the
measurements:
```bash
let augmented_data = $data
    | update label { parse "x = {x}" | into record | get x | into int } # parse the label to extract x
    | insert mean { get times | math avg }                              # compute the mean
    | insert stddev { get times | into float | math stddev }            # compute the standard deviation
```

And finally, we can plot the results:
```bash
let graphs = $augmented_data
    | rename --column { label: "x", mean: "y", stddev: "e" } # give columns the correct name
    | group-by name --to-table                               # group by the name of the graph
    | rename --column { group: name, items: "points" }

gplt plot ($graphs | to json)
```
![plnk.png](assets/plnk.png)
